<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class Harsit_Trade_Link_Payment_Gateway extends WC_Payment_Gateway {

    public function __construct() {
        $this->id                 = 'harsit_trade_link';
        $this->method_title       = 'Harsit Trade Link';
        $this->method_description = 'Pay via Harsit Trade Link';
        $this->title              = $this->get_option( 'title', 'Harsit Trade Link' );
        $this->icon               = plugins_url( 'assets/logo.png', dirname( __FILE__ ) );
        $this->has_fields         = false;

        $this->init_form_fields();
        $this->init_settings();

        add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, [ $this, 'process_admin_options' ] );
        
        // ✅ Correct API callback hook
        add_action( 'woocommerce_api_' . $this->id, [ $this, 'check_response' ] );

        if ( is_admin() && $this->get_option( 'mode' ) === 'test' ) {
            add_action( 'admin_notices', function() {
                echo '<div class="notice notice-warning"><p><strong>Harsit Payment Gateway is in TEST mode</strong>. Do not use for live payments.</p></div>';
            });
        }
    }

    public function get_description() {
        return $this->get_option( 'description' );
    }

    public function init_form_fields() {
        $this->form_fields = [
            'enabled' => [
                'title'   => 'Enable / Disable',
                'type'    => 'checkbox',
                'label'   => 'Enable Harsit Trade Link',
                'default' => 'yes',
            ],
            'title' => [
                'title'       => 'Title',
                'type'        => 'text',
                'description' => 'Title seen by users during checkout.',
                'desc_tip'    => true,
                'default'     => 'Harsit Trade Link',
            ],
            'description' => [
                'title'       => 'Description',
                'type'        => 'textarea',
                'description' => 'This controls the description which the user sees during checkout.',
                'default'     => 'Pay securely via Harsit Trade Link.',
            ],
            'transaction_prefix' => [
                'title'       => 'Transaction Prefix',
                'type'        => 'text',
                'description' => 'Custom prefix for transaction UUIDs (e.g., ht_, order_, etc.)',
                'desc_tip'    => true,
                'default'     => 'ht_',
            ],
            'mode' => [
                'title'       => 'Payment Mode',
                'type'        => 'select',
                'description' => 'Test or Live environment',
                'desc_tip'    => true,
                'default'     => 'test',
                'options'     => [
                    'test' => 'Test (UAT)',
                    'live' => 'Live (Production)',
                ],
            ],
            'secret_key' => [
                'title'       => 'API Secret Key',
                'type'        => 'text',
                'description' => 'Secret key from Harsit Trade Link',
                'desc_tip'    => true,
                'default'     => '',
            ],
            'return_url' => [
                'title'       => 'Custom Return URL',
                'type'        => 'text',
                'description' => 'Custom return URL. Leave blank to use default.',
                'default'     => '',
            ],

            'order_status' => [
                'title'       => 'Order Status on Success',
                'type'        => 'select',
                'description' => 'Select the status to set after successful payment.',
                'default'     => 'processing',
                'desc_tip'    => true,
                'options'     => [
                    'processing' => 'Processing',
                    'completed'  => 'Completed',
                ],
            ],
        ];
    }

    public function admin_options() {
        echo '<h3>' . esc_html( $this->get_method_title() ) . '</h3>';
        echo '<p>' . esc_html( $this->get_method_description() ) . '</p>';
        echo '<table class="form-table">';
        $this->generate_settings_html();
        echo '</table>';
    }

    public function process_payment( $order_id ) {
        $order              = wc_get_order( $order_id );
        $amount             = $order->get_total();
         $prefix            = $this->get_option( 'transaction_prefix', 'ht_' );
        $transaction_uuid   = uniqid( $prefix );
        $mode               = $this->get_option( 'mode' );
        $secret_key         = $this->get_option( 'secret_key' );

        // ✅ Correct return URL based on this gateway's ID
    
        $return_url = trim( $this->get_option( 'return_url' ) );
        if ( empty( $return_url ) ) {
            $return_url = WC()->api_request_url( $this->id ); // This will be: https://yourdomain.com/wc-api/harsit_trade_link
        }
        require_once plugin_dir_path( __DIR__ ) . 'includes/class-harsit-api-handler.php';

        // require_once plugin_dir_path( __FILE__ ) . '/class-harsit-api-handler.php';
        $api  = new Harsit_API_Handler( $mode );
        $res  = $api->initialize_payment( $amount, $transaction_uuid, $return_url, $secret_key );

        $order->update_meta_data( 'ht_transaction_uuid', $transaction_uuid );
        $order->save();

        if ( isset( $res['success'] ) && $res['success'] === false ) {
            wc_add_notice( 'Payment error: ' . ( $res['message'] ?? 'Unknown error' ), 'error' );
            return [ 'result' => 'fail' ];
        }

        if ( ! empty( $res['payment_url'] ) ) {
            return [
                'result'   => 'success',
                'redirect' => esc_url_raw( $res['payment_url'] ),
            ];
        } else {
            wc_add_notice( 'No payment URL returned from gateway.', 'error' );
            return [ 'result' => 'fail' ];
        }
    }

    public function check_response() {
        $txn_id        = sanitize_text_field( $_GET['txnId'] ?? '' );
        $returned_uuid = sanitize_text_field( $_GET['transaction_uuid'] ?? '' );
        $status_param  = sanitize_text_field( $_GET['status'] ?? '' );

        if ( empty( $returned_uuid ) ) {
            wc_add_notice( 'Missing transaction UUID.', 'error' );
            wp_redirect( wc_get_checkout_url() );
            exit;
        }

        $orders = wc_get_orders([
            'limit'        => 1,
            'meta_key'     => 'ht_transaction_uuid',
            'meta_value'   => $returned_uuid,
            'meta_compare' => '=',
        ]);

       if ( empty( $orders ) || ! is_array( $orders ) ) {
            wc_add_notice( 'Order not found. Payment verification failed.', 'error' );
            wp_redirect( wc_get_checkout_url() );
            exit;
        }

        $order = $orders[0];

        if ( $status_param === 'fail' ) {
            $order->update_status( 'failed', 'User failed or cancelled the payment.' );
            wc_add_notice( 'Payment failed or cancelled.', 'error' );
            wp_redirect( wc_get_checkout_url() );
            exit;
        }

        $api = new Harsit_API_Handler( $this->get_option( 'mode' ) );
        $status = $api->check_payment_status( $txn_id, $this->get_option( 'secret_key' ) );

        if ( is_array($status) && isset( $status['status'] ) && $status['status'] === 'success' ) {
            $order->payment_complete( $txn_id );
            $order->add_order_note( 'Payment verified via Harsit Trade Link. TXN: ' . $txn_id );

            $chosen_status = $this->get_option( 'order_status', 'processing' );

            if ( $chosen_status === 'completed' ) {
                $order->update_status( 'completed', 'Order marked completed after successful payment.' );
            } else {
                $order->update_status( 'processing', 'Order marked processing after successful payment.' );
            }

            wp_redirect( $this->get_return_url( $order ) );
            die();
        } else {
            $order->update_status( 'failed', 'Payment verification failed.' );
            wc_add_notice( 'Payment failed. Please try again.', 'error' );
            wp_redirect( wc_get_checkout_url() );
            die();
        }
    }
}

// ✅ Styling for payment option
add_action( 'wp_head', function() {
    if ( is_checkout() ) {
        echo '<style>
            .payment_method_harsit_trade_link img {
                max-height: 32px !important;
                width: auto !important;
            }
        </style>';
    }
});
